library(nloptr)
library(tidyverse)
library(ggplot2)
library(PSweight)
library(nleqslv)
library(R.utils)
library(ggpattern)

set.seed(19248)

## true parameters
true_p <- c(0.5,0.5)

## treatment effect in each subgroup's arm
mu1.t <- 4  
mu1.c <- 2  
sd1.t <- 2.5  
sd1.c <- 1.5 

mu2.t <- 1 
mu2.c <- 4    
sd2.t <- 1.2
sd2.c <- 3.5 

tau_true <- c(2,-3)
tau_true_all <- 2*0.5+(-3)*0.5

GenX <- function(n){
  X <- sample(1:2, size = n, replace = TRUE, prob = true_p)
  return(X)
}

GenS <- function(X){
  
  S1 <- (X ==1)
  S2 <- (X==2)
  
  S <- cbind(S1,S2)
  
  colnames(S) <- c("A","B")
  
  return(S)
}


GenY <- function(n,X,Tr,S){
  
  Y <- NULL
  
  for(i in 1:n){
    
    # treatment arm
    if(Tr[i]==1){
      if(S[i,1]){
        Y[i] <- rnorm(1,mu1.t,sd1.t)
      }else{
        Y[i] <- rnorm(1,mu2.t,sd2.t)
      }
    }
    
    
    # control arm
    if(Tr[i]==0){
      if(S[i,1]){
        Y[i] <- rnorm(1,mu1.c,sd1.c)
      }else{
        Y[i] <- rnorm(1,mu2.c,sd2.c)
      }
    }
    
  }
  
  return(Y)
}



SubAlloc <- function(tau,sigma1_vec, sigma0_vec, p_vec, N_t,
                     envy_c,lower_c,upper_c){
  
  sigma11 <- sigma1_vec[1]
  sigma21 <- sigma1_vec[2]
  
  sigma10 <- sigma0_vec[1]
  sigma20 <- sigma0_vec[2]
  
  p1 <- p_vec[1]
  p2 <- p_vec[2]
  
  # objective function
  eval_f0 <- function(x){
    # res <- 1/2*(sigma11^2/x[1]+sigma10^2/(1-x[1])) + 
    #   1/2*(sigma21^2/x[2]+sigma20^2/(1-x[2]))
    res <- p1*(sigma11^2/x[1]+sigma10^2/(1-x[1])) + 
      p2*(sigma21^2/x[2]+sigma20^2/(1-x[2]))
    return( res )
  }
  
  delta_N <- sqrt(log(N_t)/N_t)
  
  # constraint function
  eval_g0 <- function(x) {
    constr <- c(x[1]-x[2]-envy_c,
                -envy_c-x[1]+x[2],
                -delta_N-log(x[1]/(1-x[1]))*tau[1],
                -delta_N-log(x[2]/(1-x[2]))*tau[2]
                #-log(x[1]/(1-x[1]))*tau[1],
                #-log(x[2]/(1-x[2]))*tau[2]
                )
    return( constr )
  }
  
  # Solve using NLOPT_LN_COBYLA without gradient information
  res1 <- nloptr( x0=c(0.01,0.01),
                  eval_f=eval_f0,
                  lb = c(lower_c,lower_c),
                  ub = c(upper_c,upper_c),
                  eval_g_ineq = eval_g0,
                  opts = list("algorithm"="NLOPT_LN_COBYLA",
                              "xtol_rel"=1.0e-8))
  
  # optimal e*
  e_star <- res1$solution[1:2]
  
  names(e_star) <- c("A","B")
  
  return(e_star)
}



## Monte Carlo samples
Sim <- function(m,n,envy_c){
  
  # First stage
  X_1 <- GenX(m)
  S_1 <- GenS(X_1)
  
  # Stage 1: Assign treatment randomly
  T_1 <- rbinom(m,1,0.5)
  # Generate Y_1
  Y_1 <-  GenY(m,X_1,T_1,S_1)
  e_1 <- rep(1/2, 2) # randomly assign treatment with e=1/2
  p_1 <- colSums(S_1)/m
  
  # estimate subgroup ATE: use PSWeight
  tau_1 <- sd_1.t <- sd_1.c <- NULL
  
  for (j in 1:ncol(S_1)){
    dat1 <- as.data.frame(cbind(Y_1[S_1[,j]],T_1[S_1[,j]],X_1[S_1[,j]]))
    names(dat1) <- c("Y","Tr","X")
    
    tau_1[j] <-  sum(dat1$Y*dat1$Tr)/sum(dat1$Tr) - sum(dat1$Y*(1-dat1$Tr))/sum(1-dat1$Tr)
    sd_1.t[j]<- sd(dat1$Y[dat1$Tr==1])
    sd_1.c[j]<- sd(dat1$Y[dat1$Tr==0])
  }
  
  names(tau_1) <- c("A","B")
  names(sd_1.t) <- names(sd_1.c) <-  c("A","B")
  
  tau_old <- tau_1
  sd_old.t <- sd_old_db.t <- sd_1.t
  sd_old.c <- sd_old_db.c <- sd_1.c
  S_old <- S_old.db <- S_1
  n_old <- m
  T_old <- T_old.cr <- T_old.db <- T_1
  X_old <- X_1
  Y_old <- Y_old.cr <- Y_old.db <- Y_1
  
  p_old <- p_1
  
  ## fully sequential
  for(i in 1:n){ # start of fully sequential assignments
    
    X_i <- GenX(1)
    S_i <- GenS(X_i)
    S_new <- rbind(S_old,S_i)
    n_new <- n_old +1
    
    ## (1) Proposed design --------------------------------
    
    group_name <- colnames(S_i)[S_i]
    
    e_opt <- SubAlloc(tau_old,sd_old.t,sd_old.c, p_old,n_old,
                      envy_c,lower_c=0.01,upper_c=0.99)
    
    e_opt_i <- e_opt[group_name] # allocation for subject i
    
    # current allocation
    e_current_i <- sum(T_old[S_old[,group_name]])/length(T_old[S_old[,group_name]])
    
    T_i <- ifelse(e_current_i < e_opt_i,1,0)
    
    Y_i <-  GenY(1,X_i,T_i,S_i) # observe outcome of subject i
    
    # update old info
    T_old <- c(T_old, T_i)
    Y_old <- c(Y_old, Y_i)
    S_old <- S_new
    X_old <- c(X_old, X_i)
    n_old <- n_new
    
    # update tau and sd
    tau_old <- sd_old.t <-sd_old.c <- NULL
    for (j in 1:ncol(S_old)){
      dat1 <- as.data.frame(cbind(Y_old[S_old[,j]],T_old[S_old[,j]],X_old[S_old[,j]]))
      names(dat1) <- c("Y","Tr","X")
      ps <- sum(dat1$Tr)/nrow(dat1)
      
      tau_old[j] <-  sum(dat1$Y*dat1$Tr)/sum(dat1$Tr) - sum(dat1$Y*(1-dat1$Tr))/sum(1-dat1$Tr)
      sd_old.t[j]<- sd(dat1$Y[dat1$Tr==1])
      sd_old.c[j]<- sd(dat1$Y[dat1$Tr==0])
    }
    
    names(tau_old) <- c("A","B")
    names(sd_old.t) <- names(sd_old.c)<- c("A","B")
    
    
    sub_alloc_opt <- c(sum(T_old[S_old[,1]])/sum(S_old[,1]),
                       sum(T_old[S_old[,2]])/sum(S_old[,2]))
    
    ## (2) Complete randomization--------------------------------
    e_cr <- c(1/2,1/2)
    names(e_cr) <- c('A','B')
    T_i.cr <- rbinom(1,1,1/2)
    Y_i.cr <-  GenY(1,X_i,T_i.cr,S_i)
    # update old info
    T_old.cr <- c(T_old.cr, T_i.cr)
    Y_old.cr <- c(Y_old.cr, Y_i.cr)
    
    # update tau and sd
    tau_old.cr <- sd_old.cr.t <- sd_old.cr.c <- NULL
    for (j in 1:ncol(S_old)){
      dat1 <- as.data.frame(cbind(Y_old.cr[S_old[,j]],T_old.cr[S_old[,j]],X_old[S_old[,j]]))
      names(dat1) <- c("Y","Tr","X")
  
      tau_old.cr[j] <-  sum(dat1$Y*dat1$Tr)/sum(dat1$Tr) - sum(dat1$Y*(1-dat1$Tr))/sum(1-dat1$Tr)
      sd_old.cr.t[j]<- sd(dat1$Y[dat1$Tr==1])
      sd_old.cr.c[j]<- sd(dat1$Y[dat1$Tr==0])
    }
    
    names(tau_old.cr) <- c("A","B")
    names(sd_old.cr.t) <- names(sd_old.cr.c) <-  c("A","B")
    
    sub_alloc_cr <- c(sum(T_old.cr[S_old[,1]])/sum(S_old[,1]),
                      sum(T_old.cr[S_old[,2]])/sum(S_old[,2]))
   
    ## (3) DBCD --------------------------------------------
    e_db <- sd_old_db.t/(sd_old_db.t + sd_old_db.c)
    e_db_i <- e_db[group_name] # allocation for subject i
    
    # current allocation
    e_db_current_i <- sum(T_old.db[S_old.db[,group_name]])/length(T_old.db[S_old.db[,group_name]])
    
    T_i.db <- ifelse(e_db_current_i < e_db_i,1,0)
    
    Y_i.db <-  GenY(1,X_i,T_i.db,S_i) # observe outcome of subject i
    
    # update old info
    T_old.db <- c(T_old.db, T_i.db)
    Y_old.db <- c(Y_old.db, Y_i.db)
    S_old.db <- S_new
    
    # update tau and sd
    tau_old.db <- sd_old_db.t <-sd_old_db.c <- NULL
    for (j in 1:ncol(S_old)){
      dat1 <- as.data.frame(cbind(Y_old.db[S_old[,j]],T_old.db[S_old[,j]],X_old[S_old[,j]]))
      names(dat1) <- c("Y","Tr","X")
      
      tau_old.db[j] <-  sum(dat1$Y*dat1$Tr)/sum(dat1$Tr) - sum(dat1$Y*(1-dat1$Tr))/sum(1-dat1$Tr)
      sd_old_db.t[j]<- sd(dat1$Y[dat1$Tr==1])
      sd_old_db.c[j]<- sd(dat1$Y[dat1$Tr==0])
    }
    
    names(tau_old.db) <- c("A","B")
    names(sd_old_db.t) <- names(sd_old_db.c)<- c("A","B")
    
    sub_alloc_db <- c(sum(T_old.db[S_old[,1]])/sum(S_old[,1]),
                      sum(T_old.db[S_old[,2]])/sum(S_old[,2]))
    
    
  } # end of fully sequential assignments
  
  
  
  p_old <- colSums(S_old)/(nrow(S_old))

  ## Proposed design ------------------------------------
  tau_opt <- tau_old
  ate_opt <- tau_opt[1]*p_old[1] + tau_opt[2]*p_old[2]
  nu_opt <- 1/p_old*(sd_old.t^2/sub_alloc_opt + sd_old.c^2/(1-sub_alloc_opt))
  var_opt <-  sum(p_old^2*nu_opt) + (p_old[1]*(tau_opt[1]- ate_opt)^2 + p_old[2]*(tau_opt[2]- ate_opt)^2)
  sd_opt <- sqrt(var_opt)

  hi_opt <- ate_opt + 1.96*sd_opt/sqrt(m+n)
  lo_opt <- ate_opt - 1.96*sd_opt/sqrt(m+n)
  cover_opt <- (lo_opt < tau_true_all) & (hi_opt > tau_true_all)
  power_opt <- (lo_opt>0)
  
  
  ## Complete randomization ------------------------------------
  tau_cr <- tau_old.cr
  ate_cr <- tau_cr[1]*p_old[1] + tau_cr[2]*p_old[2]
  nu_cr <- 1/p_old*(sd_old.cr.t^2/sub_alloc_cr + sd_old.cr.c^2/(1-sub_alloc_cr))
  var_cr <-  sum(p_old^2*nu_cr) + (p_old[1]*(tau_cr[1]- ate_cr)^2 + p_old[2]*(tau_cr[2]- ate_cr)^2)
  sd_cr <- sqrt(var_cr)
  
  hi_cr <- ate_cr + 1.96*sd_cr/sqrt(m+n)
  lo_cr <- ate_cr - 1.96*sd_cr/sqrt(m+n)
  cover_cr <- (lo_cr < tau_true_all) & (hi_cr > tau_true_all)
  power_cr <- (lo_cr>0)
  
  ## DBCD ------------------------------------
  tau_db <- tau_old.db
  ate_db <- tau_db[1]*p_old[1] + tau_db[2]*p_old[2]
  nu_db <- 1/p_old*(sd_old_db.t^2/sub_alloc_db + sd_old_db.c^2/(1-sub_alloc_db))
  var_db <-  sum(p_old^2*nu_db) + (p_old[1]*(tau_db[1]- ate_db)^2 + p_old[2]*(tau_db[2]- ate_db)^2)
  sd_db <- sqrt(var_db)
  
  hi_db <- ate_db + 1.96*sd_db/sqrt(m+n)
  lo_db <- ate_db - 1.96*sd_db/sqrt(m+n)
  cover_db <- (lo_db < tau_true_all) & (hi_db > tau_true_all)
  power_db <- (lo_db>0)
  
  return(c(tau_opt,tau_cr,tau_db,
           sub_alloc_opt,sub_alloc_cr,sub_alloc_db,
           ate_opt,ate_cr,ate_db,
           cover_opt,cover_cr,cover_db,
           power_opt,power_cr,power_db))
}


